export type FaustModuleFactory = EmscriptenModuleFactory<FaustModule>;

export interface FaustModule extends EmscriptenModule {
    ccall: typeof ccall;
    cwrap: typeof cwrap;
    UTF8ArrayToString(u8Array: number[], ptr: number, maxBytesToRead?: number): string;
    stringToUTF8Array(str: string, outU8Array: number[], outIdx: number, maxBytesToWrite: number): number;
    UTF8ToString: typeof UTF8ToString;
    UTF16ToString: typeof UTF16ToString;
    UTF32ToString: typeof UTF32ToString;
    stringToUTF8: typeof stringToUTF8;
    stringToUTF16: typeof stringToUTF16;
    stringToUTF32: typeof stringToUTF32;
    allocateUTF8: typeof allocateUTF8;
    lengthBytesUTF8: typeof lengthBytesUTF8;
    lengthBytesUTF16: typeof lengthBytesUTF16;
    lengthBytesUTF32: typeof lengthBytesUTF32;
    FS: typeof FS;
    libFaustWasm: new () => LibFaustWasm;
}

export type FaustInfoType = "help" | "version" | "libdir" | "includedir" | "archdir" | "dspdir" | "pathslist";

export interface IntVector {
    size(): number;
    get(i: number): number;
    delete(): void;
}

export interface FaustDspWasm {
    /* The C++ factory pointer as in integer */
    cfactory: number;
    /* The compiled wasm binary code */
    data: IntVector;
    /* The DSP JSON description */
    json: string;
}

export interface LibFaustWasm {
    /**
     * Return the Faust compiler version.
     * 
     * @returns the version
     */
    version(): string;

    /**
     * Create a dsp factory from Faust code.
     *
     * @param name - an arbitrary name for the Faust module
     * @param code - Faust dsp code
     * @param args - the compiler options
     * @param useInternalMemory - tell the compiler to generate static embedded memory or not
     * @returns an opaque reference to the factory
     */
    createDSPFactory(name: string, code: string, args: string, useInternalMemory: boolean): FaustDspWasm;

    /**
     * Delete a dsp factory.
     *
     * @param cFactory - the factory C++ internal pointer as a number
     */
    deleteDSPFactory(cFactory: number): void;

    /**
     * Expand Faust code i.e. linearize included libraries.
     *
     * @param name - an arbitrary name for the Faust module
     * @param code - Faust dsp code
     * @param args - the compiler options
     * @returns return the expanded dsp code
     */
    expandDSP(name: string, code: string, args: string): string;

    /**
     * Generates auxiliary files from Faust code. The output depends on the compiler options.
     *
     * @param name - an arbitrary name for the faust module
     * @param code - Faust dsp code
     * @param args - the compiler options
     */
    generateAuxFiles(name: string, code: string, args: string): boolean;

    /**
     * Delete all existing dsp factories.
     */
    deleteAllDSPFactories(): void;

    /**
     * Exception management: gives an error string
     */
    getErrorAfterException(): string;

    /**
     * Exception management: cleanup
     * Should be called after each exception generated by the LibFaust methods.
     */
    cleanupAfterException(): void;

    /**
     * Get info about the embedded Faust engine
     * @param what - the requested info
     */
    getInfos(what: FaustInfoType): string;
}

export interface FaustDspFactory extends Required<LooseFaustDspFactory> { }

/**
 * The Factory structure.
 */
export interface LooseFaustDspFactory {
    /** a "pointer" (as an integer) on the internal C++ factory */
    cfactory?: number;
    /** the WASM code as a binary array */
    code?: Uint8Array;
    /** the compule WASM module */
    module: WebAssembly.Module;
    /** the compiled DSP JSON description */
    json: string;
    /** whether the factory is a polyphonic one or not */
    poly?: boolean;
    /** a unique identifier */
    shaKey?: string;
    /** a map of transferable audio buffers for the `soundfile` function */
    soundfiles?: Record<string, AudioData | null>;
}

export interface FaustDspMeta {
    name: string;
    filename: string;
    compile_options: string;
    include_pathnames: string[];
    inputs: number;
    outputs: number;
    size: number;
    version: string;
    library_list: string[];
    meta: { [key: string]: string }[];
    ui: FaustUIDescriptor;
}

export type FaustUIDescriptor = FaustUIGroup[];
export type FaustUIItem = FaustUIInputItem | FaustUIOutputItem | FaustUIGroup;
export interface FaustUIInputItem {
    type: FaustUIInputType;
    label: string;
    address: string;
    url: string;
    index: number;
    init?: number;
    min?: number;
    max?: number;
    step?: number;
    meta?: FaustUIMeta[];
}
export interface FaustUIOutputItem {
    type: FaustUIOutputType;
    label: string;
    address: string;
    index: number;
    min?: number;
    max?: number;
    meta?: FaustUIMeta[];
}
export interface FaustUIMeta {
    [order: number]: string;
    style?: string; // "knob" | "menu{'Name0':value0;'Name1':value1}" | "radio{'Name0':value0;'Name1':value1}" | "led";
    unit?: string;
    scale?: "linear" | "exp" | "log";
    tooltip?: string;
    hidden?: string;
    [key: string]: string | undefined;
}
export type FaustUIGroupType = "vgroup" | "hgroup" | "tgroup";
export type FaustUIOutputType = "hbargraph" | "vbargraph";
export type FaustUIInputType = "vslider" | "hslider" | "button" | "checkbox" | "nentry" | "soundfile";
export interface FaustUIGroup {
    type: FaustUIGroupType;
    label: string;
    items: FaustUIItem[];
}
export type FaustUIType = FaustUIGroupType | FaustUIOutputType | FaustUIInputType;

export interface AudioParamDescriptor {
    automationRate?: AutomationRate;
    defaultValue?: number;
    maxValue?: number;
    minValue?: number;
    name: string;
}

export interface AudioWorkletProcessor {
    port: MessagePort;
    process(inputs: Float32Array[][], outputs: Float32Array[][], parameters: Record<string, Float32Array>): boolean;
}
export declare const AudioWorkletProcessor: {
    prototype: AudioWorkletProcessor;
    parameterDescriptors: AudioParamDescriptor[];
    new(options: AudioWorkletNodeOptions): AudioWorkletProcessor;
};

export interface AudioWorkletGlobalScope {
    AudioWorkletGlobalScope: any;
    globalThis: AudioWorkletGlobalScope;
    registerProcessor: (name: string, constructor: new (options: any) => AudioWorkletProcessor) => void;
    currentFrame: number;
    currentTime: number;
    sampleRate: number;
    AudioWorkletProcessor: typeof AudioWorkletProcessor;
}

export interface InterfaceFFT {
    forward(arr: ArrayLike<number> | ((arr: Float32Array) => any)): Float32Array;
    inverse(arr: ArrayLike<number> | ((arr: Float32Array) => any)): Float32Array;
    dispose(): void;
}
export declare const InterfaceFFT: {
    new(size: number): InterfaceFFT;
}

export type TWindowFunction = (index: number, length: number, ...args: any[]) => number;
export type Writeable<T> = { -readonly [P in keyof T]: T[P] };
export type TypedArray = Int8Array | Uint8Array | Int16Array | Uint16Array | Int32Array | Uint32Array | Uint8ClampedArray | Float32Array | Float64Array;
export type TypedArrayConstructor = typeof Int8Array | typeof Uint8Array | typeof Int16Array | typeof Uint16Array | typeof Int32Array | typeof Uint32Array | typeof Uint8ClampedArray | typeof Float32Array | typeof Float64Array;

export declare const FFTUtils: {
    /** Inject window functions as array, no need to add rectangular (no windowing) */
    windowFunctions?: TWindowFunction[];
    /** Get a FFT interface constructor */
    getFFT: () => Promise<typeof InterfaceFFT>;
    /** Convert from FFTed (spectral) signal to three arrays for Faust processor's input, fft is readonly, real/imag/index length = *fftSize* / 2 + 1; fft length depends on the FFT implementation */
    fftToSignal: (fft: Float32Array | Float64Array, real: Float32Array | Float64Array, imag?: Float32Array | Float64Array, index?: Float32Array | Float64Array) => any;
    /** Convert from Faust processor's output to spectral data for Inversed FFT, real/imag are readonly, real/imag length = *fftSize* / 2 + 1; fft length depends on the FFT implementation */
    signalToFFT: (real: Float32Array | Float64Array, imag: Float32Array | Float64Array, fft: Float32Array | Float64Array) => any;
    /** Convert from Faust processor's output to direct audio output, real/imag are readonly, fft length = fftSize = (real/imag length - 1) * 2 */
    signalToNoFFT: (real: Float32Array | Float64Array, imag: Float32Array | Float64Array, fft: Float32Array | Float64Array) => any;
}

export interface AudioData {
    sampleRate: number;
    audioBuffer: Float32Array[];
}
